<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Http\Requests\Product\ProductRequest;
use App\Http\Requests\Search\SearchProductRequest;
use App\Http\Resources\ProductResource;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $perPage = \returnPerPage();
        $products = Product::paginate($perPage);
        ProductResource::collection($products);
        return \Pagination($products);
    }
    /**
     * Display a listing of special products.
     */
    public function specialIndex()
    {
        $products = Product::where('special', 1)->get();
        return \SuccessData(__('public.special_products_retrieved'), ProductResource::collection($products));
    }

    /**
     * Display a listing of the resource.
     */
    public function indexCategory(Request $request)
    {
        $perPage = \returnPerPage();
        $products = Product::where('category_id', $request->categoryId)->paginate($perPage);
        ProductResource::collection($products);
        return \Pagination($products);
    }


    /**
     * Store a newly created resource in storage.
     */
    public function store(ProductRequest $request)
    {
        try {
            $product = Arr::only($request->validated(), ['name', 'price', 'original_price', 'image', 'description', 'category_id', 'is_stock', 'discount']);
            if (!$product['is_stock']) {
                $product['is_stock'] = false;
            }
            $path = 'Images/Products/';

            if ($request->hasFile('image') && $request->file('image')->isValid()) {
                $product['image'] = \uploadImage($request->image, $path);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid image file',
                    'code' => 422,
                    'data' => null,
                ], 422);
            }

            Product::create($product);
            return \Success(__('public.product_created'));
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Product creation failed: ' . $e->getMessage(),
                'code' => 500,
                'data' => null,
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Product $product)
    {
        return \SuccessData(__('public.product_retrieved'), new ProductResource($product->load(['category', 'tags', 'productDetails'])));
    }

    public function search(SearchProductRequest $request)
    {
        $searchArr = Arr::only($request->validated(), ['name']);
        $cities = Product::where('name', 'LIKE', "%{$searchArr['name']}%")->orWhere('name', $searchArr['name'])->get();
        return \SuccessData(__('public.Show'), ProductResource::collection($cities));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(ProductRequest $request, Product $product)
    {
        $product->update($request->validated());
        return \SuccessData(__('public.product_updated'), new ProductResource($product->load(['category', 'tags', 'productDetails'])));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Product $product)
    {
        $product->delete();
        return \SuccessData(__('public.product_deleted'), null);
    }
}
