<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Product;
use App\Http\Requests\Order\OrderRequest;
use App\Http\Resources\OrderResource;

class OrderController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $orders = auth()->user()->orders()->with(['orderItems'])->orderBy('created_at', 'desc')->get();
        return \SuccessData(__('public.orders_retrieved'), OrderResource::collection($orders));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(OrderRequest $request)
    {
        $data = $request->validated();
        $data['user_id'] = auth()->id();

        $totalPrice = 0;
        $itemsWithPrices = [];
        foreach ($request->items as $item) {
            $product = Product::find($item['product_id']);
            $item['price'] = $product->price;
            $totalPrice += $item['price'] * $item['quantity'];
            $itemsWithPrices[] = $item;
        }

        $data['total_price'] = $totalPrice;
        $order = Order::create($data);

        foreach ($itemsWithPrices as $item) {
            $order->orderItems()->create($item);
        }

        return \Success(__('public.order_created'));
    }

    /**
     * Display the specified resource.
     */
    public function show(Order $order)
    {
        return \SuccessData(__('public.order_retrieved'), new OrderResource($order->load(['user', 'orderItems.product'])));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(OrderRequest $request, Order $order)
    {
        $order->update($request->validated());
        return \SuccessData(__('public.order_updated'), new OrderResource($order->load(['user', 'orderItems.product'])));
    }

    /**
     * Update the specified resource in storage.
     */
    public function cancelOrder(Order $order)
    {
        if ($order->user_id !== auth()->id()) {
            abort(403, 'Unauthorized');
        }
        $order->update(['status' => 4]);
        return \Success(__('public.order_updated'));
    }
    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Order $order)
    {
        $order->delete();
        return \SuccessData(__('public.order_deleted'), null);
    }
}
